function [qv, correctedData, integral]=integrate(data,intLimits,baselineLimits,acq_period)

%integrate subtracts a user chosen sloped or straight baseline from the raw
%gating data, integrates the gating current between two user defined
%points, and returns the QV data. It is called by gatingwrapper

%This code ensures the user has entered two limits for the integration and the
%baseline
if length(intLimits)~=2||length(baselineLimits)~=2
    error('Limit inputs must contain two values.');
end

%Initialize arrays for correctedData, the baseline, the integral of the
%data, and the QV values
correctedData=zeros(intLimits(2)-intLimits(1)+1,size(data,2));
leakOffset=zeros(intLimits(2)-intLimits(1)+1,size(data,2));
integral=zeros(intLimits(2)-intLimits(1)+1,size(data,2));
qv=zeros(size(data,2),1);

%Allow user to choose mean or slope baseline
baseline_type=menu('Choose a baseline type','Mean between two points', 'Line between two points');
if baseline_type==1
    for n=1:size(data,2)
        %Here we are just determining a single value (leakOffset)
        leakOffset(:,n)=(mean(data(baselineLimits(1):baselineLimits(2),n)));
        %We then subtract this value from the range of the gating current we
        %are going to integrate
        correctedData(:,n)=data(intLimits(1):intLimits(2),n)-leakOffset(:,n);
        %Integrate the baselined data, accounting for the acquisition rate
        %and converting uA to A
        integral(:,n)=cumsum(correctedData(:,n))*1e-12*acq_period;
        %The line below could be used for future automatic normalized values of
        %the QV, it should provide the greatest value of Q that is found.
        [cow,i]=max(abs(integral(:,n)));
        %Set qv equal to the integral of the gating current, or Q
        qv(n)=integral(i,n);
    end

else
    for n=1:size(data,2)
        %Here we are fitting a line to the gating current curve between the
        %user assigned baselineLimits
        leakfit=fit((baselineLimits(1):baselineLimits(2))',data(baselineLimits(1):baselineLimits(2),n),'poly1');
        fitCoeffs=coeffvalues(leakfit);
        %leakOffset is set to equal this curve across the range of the gating
        %current we are going to integrate
        leakOffset(:,n)=(intLimits(1):intLimits(2))*fitCoeffs(1)+fitCoeffs(2);
        %Delete leakOffset from the gating current we are going to integrate
        correctedData(:,n)=data(intLimits(1):intLimits(2),n)-leakOffset(:,n);
        %Integrate the baselined data, accounting for the acquisition rate
        %and converting uA to A
        integral(:,n)=cumsum(correctedData(:,n))*1e-12*acq_period;
        %The line below could be used for future automatic normalized values of
        %the QV, it should provide the greatest value of Q that is found.
        [cow,i]=max(abs(integral(:,n)));
        %Set qv equal to the integral of the gating current, or Q
        qv(n)=integral(i,n);
    end
end

%Assign x values for plots
x=(intLimits(1):intLimits(2))';
%Plot the raw gating currents with the lines used for baseline subtraction
figure;
plot(x,data(intLimits(1):intLimits(2),:),'b',x,leakOffset,'r');
%Plot the baseline subtracted data
figure;
plot(x,correctedData);
%Plot the integral of the gating current
figure;
plot(x,integral);
%Plot the qv
figure;
plot(qv,'o');
%output=[qv, correctedData];